"""
Blue Triangle Content Security Policy API Endpoint Example Script
Additional Documentation available in the link below
https://help.bluetriangle.com/hc/en-us/articles/360019130873-Content-Security-Policy-CSP-Endpoint

Created: March 3rd 2020
Updated: March 12th 2020
Language: Python 3.5+
Version: 1.2.0

The Content Security Policy (CSP) endpoint allows you to view a list of configured CSPs or view
an individual CSP for a specific site. Returns each CSP and version associated with the site,
in JSON format. This example script demonstrates how to retrieve this list of CSP from the API,
and list each CSP Name and Version available, or also simply print the CSP or the CSP meta-tag.
The required email address and key can be specified as command line parameters, or entered into
the variables within the script.

usage: csp_api_example.py [-h] --prefix SITE_PREFIX
                               [--email]
                               [--apiKey]
                               [--list]
                               [--metaTag]
                               [--responseHeader]
                               [--latest]
                               [--id CSP_ID]
                               [--name CSP_NAME]
                               [--version CSP_VERSION]

required arguments:
  --prefix SITE_PREFIX

optional arguments:
  -h, --help             show this help message and exit
  --email API_EMAIL      the email address for the API request
  --apiKey API_KEY       the API Key for the API request
  --list                 list all CSP Names and CSP IDs
  --metaTag              print the CSP meta-tag for the designated CSP ID and CSP Version
  --responseHeader       print the CSP Response Header for the designated CSP ID and CSP Version
  --latest               views the latest CSP that was created
  --name CSP_NAME        specifies the name of the CSP to view
  --id CSP_ID            specifies the ID of the CSP to view
  --version CSP_VERSION  specifies the version of the CSP to view

examples:

  1) Lists each CSP Name, CSP Version, and CSP ID
     csp_api_example.py --prefix mysite --list

  2) Lists the latest CSP Name, CSP Version, and CSP ID via a specific API Email and API Key
     csp_api_example.py --email example@bluetriangletech.com --apiKey 12345 --prefix mysite --list --latest

  3) Prints the CSP MetaTag for the latest CSP
     csp_api_example.py --prefix mysite --latest --metaTag

  4) Prints the CSP MetaTag for the specified CSP ID and CSP Version
     csp_api_example.py --prefix mysite --metaTag --id 12345 --version 1

  5) Prints the CSP MetaTag for the specified CSP Name and CSP Version
     csp_api_example.py --prefix mysite --metaTag --name "Example CSP" --version 1

  6) Prints the CSP Response Header (CSP Only) for the specified CSP ID and CSP Version
     csp_api_example.py --prefix mysite --responseHeader --id 12345 --version 1

  7) Prints the CSP Response Header (CSP Only) for the specified CSP Name and CSP Version
     csp_api_example.py --prefix mysite --responseHeader --name "Example CSP" --version 1

  8) Prints the CSP MetaTag for the latest version of the specified CSP ID
     csp_api_example.py --prefix mysite --metaTag --id 12345 --version latest
"""

import requests
import json
import argparse
import sys

# Required configurations for the Blue Triangle API
# API Email and Key are available in the Blue Triangle Portal
# Under the User Menu -> View Profile page
# The API Email and Key can also be passed through the command line options
# --email example@bluetriangletech.com
# --apiKey 12345
BTT_CSP_API_URL = 'https://api.bluetriangletech.com/content-security-policies'
BTT_X_API_EMAIL = 'example@bluetriangletech.com'
BTT_X_API_KEY = '12345'
API_TIMEOUT_SECONDS = 30

# For parsing command line arguments
parser = argparse.ArgumentParser(description='Blue Triangle Content Security Policy API Endpoint Example Script')
parser.add_argument('--prefix', required=True, dest="site_prefix", type=str)
parser.add_argument('--name', dest='csp_name', type=str)
parser.add_argument('--id', dest="csp_id", type=str)
parser.add_argument('--version', dest='csp_version', type=str)
parser.add_argument('--list', action="store_true", dest="list_csp_only", default=False)
parser.add_argument('--metaTag', action="store_true", dest="csp_meta_tag", default=False)
parser.add_argument('--responseHeader', action="store_true", dest="csp_response_header", default=False)
parser.add_argument('--latest', action="store_true", dest="show_only_latest", default=False)
parser.add_argument('--email', dest="btt_api_email", type=str)
parser.add_argument('--apiKey', dest="btt_api_key", type=str)

# Store command line arguments
arg_results = parser.parse_args()
site_prefix = arg_results.site_prefix
csp_name = arg_results.csp_name
csp_id = arg_results.csp_id
csp_version = arg_results.csp_version
list_csp_only = arg_results.list_csp_only
csp_meta_tag = arg_results.csp_meta_tag
csp_response_header = arg_results.csp_response_header
btt_api_email = arg_results.btt_api_email
btt_api_key = arg_results.btt_api_key
show_only_latest = arg_results.show_only_latest

# Overwrites the API Email and Key if specified on the command line
if btt_api_email is not None:
    BTT_X_API_EMAIL = btt_api_email
if btt_api_key is not None:
    BTT_X_API_KEY = btt_api_key

# Request Header fields for the Blue Triangle API
request_headers = {
  'X-API-Email': BTT_X_API_EMAIL,
  'X-API-Key': BTT_X_API_KEY,
  'Content-Type': 'application/json'
}

# Add the site prefix to the API endpoint
btt_endpoint_url = BTT_CSP_API_URL + '?prefix=' + site_prefix

# If the latest flag is specified
if show_only_latest is True:
    btt_endpoint_url += '&version=latest'

if csp_name is not None:
    btt_endpoint_url += '&name=' + csp_name

if csp_id is not None:
    btt_endpoint_url += '&id=' + csp_id

if show_only_latest is False and csp_version is not None:
    btt_endpoint_url += '&version=' + csp_version

api_response = None
try:
    api_response = requests.request("GET", btt_endpoint_url, headers=request_headers, timeout=API_TIMEOUT_SECONDS)
    api_response.raise_for_status()
except requests.exceptions.ConnectionError as conn_error:
    print("Connection Error:", conn_error)
except requests.exceptions.Timeout as timeout_error:
    print("Timeout Error:", timeout_error)
except requests.exceptions.HTTPError as http_error:
    print("HTTP Error:", http_error)
except requests.exceptions.RequestException as req_error:
    print("Request Error:", req_error)

if api_response is None:
    print("Unable to access API Response, exiting...")
    sys.exit()

# Parse the JSON response from the API into a Dictionary Object
try:
    csp_data_dict = json.loads(api_response.text)
except ValueError:
    print("Error decoding API Response JSON, exiting...")
    sys.exit()

# Command line option to list the CSP Name, CSP Version, and CSP ID
# Showing only the latest will have only one CSP returned
if list_csp_only is True and show_only_latest:
    print("CSP Name: '%s', CSP Version: %s, CSP ID: %s" % (csp_data_dict['name'], csp_data_dict['version'], csp_data_dict['id']))
elif list_csp_only is True:
    for csp in csp_data_dict:
        print("CSP Name: '%s', CSP Version: %s, CSP ID: %s" % (csp['name'], csp['version'], csp['id']))

# If the metaTag or responseHeader options are specified,
# then either the name/id must be given with the version
if show_only_latest and csp_meta_tag:
    print(csp_data_dict['metaTag'])
elif show_only_latest and csp_response_header:
    print(csp_data_dict['responseHeader'])
elif csp_meta_tag is True:
    for csp in csp_data_dict:
        if csp_id == csp['id'] and csp_version == csp['version']:
            print(csp['metaTag'])
        elif csp_name == csp['name'] and csp_version == csp['version']:
            print(csp['metaTag'])
elif csp_response_header is True:
    for csp in csp_data_dict:
        if csp_id == csp['id'] and csp_version == csp['version']:
            print(csp['responseHeader'])
        elif csp_name == csp['name'] and csp_version == csp['version']:
            print(csp['responseHeader'])
